# Copyright (c) 2015 - 2017 Jonas Peters  [jonas.peters@math.ku.dk] and Nicolai Meinshausen [meinshausen@gmail.com]
# All rights reserved.  See the file COPYING for license terms. 
## Function completing parameter list for useIda with default values
## @author Alain Hauser
idaPars <- function(useThoseIntervsForTraining = NA,
                    useThoseIntervsForTesting = NA, 
                    numCores = 1, 
                    alpha = 0.01, 
                    makeNewComputation = FALSE,
                    usePCSkeleton = FALSE,
                    pcFileName = NULL)
{
    list(useThoseIntervsForTraining = useThoseIntervsForTraining, 
         useThoseIntervsForTesting = useThoseIntervsForTesting,
         numCores = numCores, 
         alpha = alpha, 
         makeNewComputation = makeNewComputation,
         usePCskeleton = usePCSkeleton,
         pcFileName = pcFileName)
}

useIda <- function(dataFileName,
                   method,
                   graphFileName,
                   idaFileName,
                   #intTrain = integer(0),  #NOT USED ANYMORE
                   #intTest = 1:data$nInt,
                   pars = idaPars())
{
    # Alain Hauser + Jonas Peters + Joris Mooij
    # 2014-03-13 + 2014-06-16
    # method can be 
    #       PC, PCstablefast, PCstable
    #       GES, empty, CAM, random
    
    library(pcalg)
    library(Matrix)
    
    load(dataFileName)
    p <- data$p
    
    
    computeGraph <- !file.exists(graphFileName)
    if(pars$makeNewComputation)
        computeGraph <- TRUE
    
    #    if(method == "random")
    #    {
    #        counterForRandom <- 1
    #        while(file.exists(paste(graphFileName,counterForRandom,sep = "")))
    #        {
    #            counterForRandom <- counterForRandom + 1
    #        }
    #    }
    
    #####
    # computing graph 
    #####
    if(computeGraph)
    {
        if( method == 'PC' || method == 'PCstablefast' || method == 'PCstable' ) 
        {
            ## TODO: Currently ignoring pars$useThoseIntervsForTraining... Check: should we include 
            ## interventional training samples and handle them as observational ones?
            ## Same question for GES
            print('Starting pc, be patient!')
            print(Sys.time())
            skel.method <- switch(method,
                                  PC = 'original',
                                  PCstablefast = 'stable.fast',
                                  PCstable = 'stable')
            pc.fit <- pc(suffStat=list(C=cor(data$obs),n=data$nObs),skel.method=skel.method,indepTest=gaussCItest,p=p,alpha=pars$alpha)
            print(Sys.time())
            Adj <- pc.fit@graph
            save(Adj,file=graphFileName)
            # 2.7 h on hactar, 220 MB file
        } else if( method == 'empty' ) 
        {
            Adj <- matrix(0,p,p)
            Adj <- as(Adj, "graphNEL")
            save(Adj,file=graphFileName)
        } else if( method == 'random' ) 
        {
            cat("generating random DAG...\n")
            Adj <- genRandomDAG(p,pars$probConnect)
            Adj <- as(Adj, "graphNEL")
            cat("DONE!\n")
            save(Adj,file=graphFileName)
            #            save(Adj,file=paste(graphFileName,counterForRandom, sep = ""))
        } else if( method == 'trueDAG' )
        {
            cat("Using true DAG...\n")
            if( dim(data$B)[1] != p || dim(data$B)[2] != p )
                error('data$B should have size p x p')
            Adj <- as(sign(abs(data$B)),'graphNEL')
            save(Adj,file=graphFileName)
        } else if( method == 'trueCPDAG' )
        {
            cat("Using true CPDAG...\n")
            if( dim(data$B)[1] != p || dim(data$B)[2] != p )
                error('data$B should have size p x p')
            Adj <- as(dag2cpdag(as(sign(abs(data$B)),'graphNEL')),'graphNEL')
            save(Adj,file=graphFileName)
        } else if( method == 'strongCor' ) 
        {
            C <- cor(data$obs)
            C[matrix(c(1:p,1:p),p,2)] <- 0
            C <- abs(C)
            Adj <- matrix(0,p,p)
            k <- 2
            for( i in 1:p )
                Adj[i,order(C[i,])[(p-k+1):p]] <- 1
            Adj <- ((Adj + t(Adj)) != 0)
            Adj <- as(Adj, 'graphNEL')
            #	Adj <- as((abs(C) > 0.7), 'graphNEL')
            save(Adj,file=graphFileName)
        } else if( method == 'CAM' ) 
        {
            fitCAMModel(dataFileName,graphFileBeforePruning = pars$graphFileBeforePruning, graphFileAfterPruning= graphFileName, pns.size= pars$pns.size, pruning.pval=pars$pruning.pval)
        } else if( method == 'GES' ) 
        {
            score <- new("GaussL0penObsScore", data$obs)
            if(pars$usePCSkeleton)
            {
                computeSkel <- !file.exists(pars$pcFileName)
                if(computeSkel)
                {
                    show("Computing PC...")
                    skel.fit <- skeleton(suffStat = list(C = cor(data$obs), n = data$nObs),
                                         method = 'stable.fast',
                                         indepTest = gaussCItest, 
                                         p = p,
                                         alpha = pars$alpha,
                                         numCores = pars$numCores)
                    graphObjectPC <- skel.fit@graph
                    save(graphObjectPC, file = pars$pcFileName)
                    show("DONE")
                }
                load(pars$pcFileName)
                
                skel <- as(graphObjectPC, "matrix")
                fixedGaps <- !skel
            } else {
                fixedGaps <- NULL
            }
            
            show("Computing GES...")
            if (packageVersion("pcalg") >= "2.1.0") {
                ges.fit <- ges(score, fixedGaps = fixedGaps)
            } else {
                ges.fit <- ges(p, score, fixedGaps = fixedGaps)
            }
            show("DONE.")
            
            Adj <- as(ges.fit$essgraph, "graphNEL")
            save(Adj,file=graphFileName)
        } else if( method == 'GIES' )
        {
            # Create data matrix and score object
            numIntTrain <- length(pars$useThoseIntervsForTraining)
            X <- rbind(data$obs, data$int[pars$useThoseIntervsForTraining])
            knockout <- matrix(FALSE, ncol = p, nrow = data$nObs + numIntTrain)
            if (numIntTrain > 0) {
                knockout[(1 + data$nObs):(numIntTrain + data$nObs), data$intpos[pars$useThoseIntervsForTraining]] <- TRUE
            }
            targetList <- mat2targets(knockout)
            score <- new("GaussL0penIntScore", X, targetList$targets, targetList$target.index)
            
            if(pars$usePCSkeleton)
            {
                computeSkel <- !file.exists(pars$pcFileName)
                if(computeSkel)
                {
                    show("Computing PC...")
                    skel.fit <- skeleton(suffStat = list(C = cor(data$obs), n = data$nObs),
                                         method = 'stable.fast',
                                         indepTest = gaussCItest, 
                                         p = p,
                                         alpha = pars$alpha,
                                         numCores = pars$numCores)
                    graphObjectPC <- skel.fit@graph
                    save(graphObjectPC, file = pars$pcFileName)
                    show("DONE")
                }
                load(pars$pcFileName)
                
                skel <- as(graphObjectPC, "matrix")
                fixedGaps <- !skel
            } else {
                fixedGaps <- NULL
            }
            
            show("Computing GIES...")
            if (packageVersion("pcalg") >= "2.1.0") {
                gies.fit <- gies(score, fixedGaps = fixedGaps)
            } else {
                gies.fit <- gies(p, targetList$targets, score, fixedGaps = fixedGaps)
            }
            show("DONE.")
            
            Adj <- as(gies.fit$essgraph, "graphNEL")
            save(Adj,file=graphFileName)
        } else if( method == 'nicolai-newidea' )
        { 
            ##############
            # here starts nicolai's code!!!
            # Adj <- nicolaismethod(data$obs), where data$obs is an n*p matrix
            # here ends nicolai's code!!!
            ##############
            save(Adj,file=graphFileName)
            stop('Not implemented yet!')
        } else
            stop('Invalid method in useIda.R')
    }
    
    
    #####
    # loading graph
    #####
    load(graphFileName)
    
    
    #####
    # applying ida
    #####
    print('Starting IDA, be patient!')
    if(is.na(pars$useThoseIntervsForTesting)) {
        pars$useThoseIntervsForTesting = 1:nInt
    }
    tic <- proc.time()[3]
    ida.rel.predictions <- wrapIda(X=data$obs,intervNodes=data$intpos[pars$useThoseIntervsForTesting],graphObject=Adj,numCores=pars$numCores)
    show(proc.time()[3] - tic)
    
    
    #####
    # saving ida
    #####
    print('Saving results')
    #    if( method == 'random' ) 
    #        idaFileName <- paste(idaFileName,counterForRandom, sep = "")
    save(ida.rel.predictions,file=idaFileName)        
}



wrapIda <- function(X,intervNodes=1:dim(X)[2], graphObject, numCores = 1)
    # 2014, Jonas Peters, Joris Mooij
{
    p <- dim(X)[2]
    n <- dim(X)[1]
    
    ida.rel.pred <- Matrix(NA,ncol=p,nrow=length(intervNodes))
    mcov <- cov(X)
    
    if(numCores == 1)
    {
        for(j in 1:length(intervNodes)) {
            i <- intervNodes[j]
            cat("currently computing interventional effects from node ", i, " on all others (", j, "out of", length(intervNodes), ")\n")
            idaCEs <- idaFast(x.pos=i,y.pos.set=(1:p)[-i],mcov=mcov,graphEst=graphObject)
            ida.rel.pred[j,-i] <- apply(X=abs(idaCEs),MARGIN=1,FUN=min)
        }
    } else
    {
        library(foreach)
        library(doMC)
        registerDoMC(numCores)
        results = foreach(j = 1:length(intervNodes)) %dopar% {
            i <- intervNodes[j]
            cat("computing intervention effects from node ", i, " on all others (", j, "out of", length(intervNodes), ")\n")
            idaCEs <- idaFast(x.pos=i,y.pos.set=(1:p)[-i],mcov=mcov,graphEst=graphObject)
            apply(X=abs(idaCEs),MARGIN=1,FUN=min)        
        }
        # combine results
        for(j in 1:length(intervNodes)) {
            i <- intervNodes[j]
            ida.rel.pred[j,-i] <- results[[j]]
        }
    }
    
    return(ida.rel.pred)
}



genRandomDAG <- function(p,probConnect,causalOrder = sample(p,p,replace=FALSE))
    # Copyright (c) 2010 - 2012  Jonas Peters  [peters@stat.math.ethz.ch]
    #    
    # All rights reserved.  See the file COPYING for license terms. 
    #
    # simulates a directed acyclic graph (DAG) and returns its adjacency matrix
    # 
    # INPUT:
    #   p           number of nodes 
    #   probConnect the probability that an edge i -> j is added to the DAG
    #   causalOrder starting with sink node (also called topological order)
    #   
# OUTPUT:
#   DAG         Adjacency matrix of a directed acyclic graph (DAG)    
{
    #DAG <- as(diag(rep(0,p)),"sparseMatrix")
    DAG <- Matrix(nrow=p,ncol=p,0,sparse=TRUE)
    for(i in 1:(p-2))
    {
        node <- causalOrder[i]
        possibleParents <- causalOrder[(i+1):p]
        numberParents <- rbinom(n=1,size=(p-i),prob=probConnect)
        Parents <- sample(x = possibleParents, size = numberParents, replace = FALSE)
        DAG[Parents,node] <- rep(1,numberParents)
    }
    # Sample does not work properly when choosing from sets with one element. We thus consider the last case separately.  
    node <- causalOrder[p-1]
    ParentYesNo <- rbinom(n=1,size=1,prob=probConnect)
    DAG[causalOrder[p],node] <- ParentYesNo
    
    return(DAG)
}
