genRandomDAG <- function(p,probConnect,causalOrder = sample(p,p,replace=FALSE))
# Copyright (c) 2010 - 2012  Jonas Peters  [peters@stat.math.ethz.ch]
#    
# All rights reserved.  See the file COPYING for license terms. 
#
# simulates a directed acyclic graph (DAG) and returns its adjacency matrix
# 
# INPUT:
#   p           number of nodes 
#   probConnect the probability that an edge i -> j is added to the DAG
#   causalOrder starting with sink node (also called topological order)
#   
# OUTPUT:
#   DAG         Adjacency matrix of a directed acyclic graph (DAG)    
{
    #DAG <- as(diag(rep(0,p)),"sparseMatrix")
    DAG <- Matrix(nrow=p,ncol=p,0,sparse=TRUE)
    for(i in 1:(p-2))
    {
        node <- causalOrder[i]
        possibleParents <- causalOrder[(i+1):p]
        numberParents <- rbinom(n=1,size=(p-i),prob=probConnect)
        Parents <- sample(x = possibleParents, size = numberParents, replace = FALSE)
        DAG[Parents,node] <- rep(1,numberParents)
    }
    # Sample does not work properly when choosing from sets with one element. We thus consider the last case separately.  
    node <- causalOrder[p-1]
    ParentYesNo <- rbinom(n=1,size=1,prob=probConnect)
    DAG[causalOrder[p],node] <- ParentYesNo
    
    return(DAG)
}


# Copyright (c) 2013-2014  Jonas Peters  [peters@stat.math.ethz.ch]
#                          Jan Ernest    [ernest@stat.math.ethz.ch]
# All rights reserved.  See the file COPYING for license terms. 

genRandomDAG <- function(p,probConnect,causalOrder = sample(p,p,replace=FALSE))
    # Copyright (c) 2010 - 2012  Jonas Peters  [peters@stat.math.ethz.ch]
    #    
    # All rights reserved.  See the file COPYING for license terms. 
    #
    # simulates a directed acyclic graph (DAG) and returns its adjacency matrix
    # 
    # INPUT:
    #   p           number of nodes 
    #   probConnect the probability that an edge i -> j is added to the DAG
    #   causalOrder starting with sink node (also called topological order)
    #   
# OUTPUT:
#   DAG         Adjacency matrix of a directed acyclic graph (DAG)    
{
    #DAG <- as(diag(rep(0,p)),"sparseMatrix")
    DAG <- Matrix(nrow=p,ncol=p,0,sparse=TRUE)
    for(i in 1:(p-2))
    {
        node <- causalOrder[i]
        possibleParents <- causalOrder[(i+1):p]
        numberParents <- rbinom(n=1,size=(p-i),prob=probConnect)
        Parents <- sample(x = possibleParents, size = numberParents, replace = FALSE)
        DAG[Parents,node] <- rep(1,numberParents)
    }
    # Sample does not work properly when choosing from sets with one element. We thus consider the last case separately.  
    node <- causalOrder[p-1]
    ParentYesNo <- rbinom(n=1,size=1,prob=probConnect)
    DAG[causalOrder[p],node] <- ParentYesNo
    
    return(DAG)
}


randomB <- function(G,lB = 0.1,uB = 0.9,twoIntervals = 1)
    # if twoIntervals == TRUE, lB and uB should be positive
{
    numCoeff <- sum(G)
    B <- t(G)
    if(numCoeff ==1)
    {
        coeffs <- sample(c(-1,1),size=numCoeff,0.5)^(twoIntervals) * runif(1,lB,uB)
    }
    else
    {
        coeffs <- diag(sample(c(-1,1),size=numCoeff,0.5)^(twoIntervals)) %*% runif(numCoeff,lB,uB)
    }
    B[B==1] <- coeffs
    return(B)
}



sampleDataFromG <- function(n,G,funcType="GAM", parsFuncType=list(B=randomB(G),kap=1,sigmax=1,sigmay=1,output=FALSE), noiseType="normalRandomVariances", parsNoise=list(noiseExp=1,varMin=1,varMax=2,noiseExpVarMin=2,noiseExpVarMax=4,bound=rep(1,dim(G)[2])), intervMat = NA,intervValues = NA, intervData = FALSE, changeNoise = FALSE, boundsForFactorNoiseMultiplication = c(), newBeta = NA)
    # 
    # Generates n samples according to structural equation models based on the DAG G 
    # with specified function class and given noise distribution. Uses Gaussian processes to sample the 
    # nonlinear functions.
    # 
    # INPUT:
    #   m           number of samples that should be simulated
    #   G           adjacency matrix of the full DAG
    #   funcType    parameter to choose between different function types. Default is "GAM" which simulates from 
    #               an additive model (i.e. each node is an additive function of its parents). "GP" samples from  
    #               a fully non-additive function, wheras "GAMGP" interpolates between GAM and GP with parameter kap.
#               If funcType="linear" then the function samples from a linear SEM.
#   parsFuncType 
#       kap     interpolates between a general additive model and a fully nonparametric model.
#               kap == 1 --> GAM with additive noise, kap == 0 --> fully nonparametric with additive noise
#       sigmax
#       sigmay  parameters used to simulate the Gaussian kernels
#   noiseType   specifies the type of additive noise in the model. Default is "normalRandomVariances" which simulates
#               Gaussian noise with random variances. 
#   parsNoise   list of parameters to modify the noise distribution.
#   intervMat   default: NA
#   intervData  default: FALSE
#
# OUTPUT:
#   X           n x p matrix containing the n samples of the specified model.    

{
    if(funcType == "linear")
    {
        if(intervData)
        {
            sampleDataFromGLinearInterv(n=n,G=G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData, changeNoise, boundsForFactorNoiseMultiplication, newBeta)
        } else
        {
            sampleDataFromGLinear(n=n,G=G,parsFuncType,noiseType,parsNoise)            
        }
        
    } else if(funcType == "GAM") 
    {
        parsFuncType$kap = 1
        sampleDataFromGAMGP(n=n,G=G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData)
        
    } else if(funcType == "GP")
    {
        parsFuncType$kap = 0
        sampleDataFromGAMGP(n=n,G=G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData)
        
    } else if(funcType == "GAMGP")
    {
        sampleDataFromGAMGP(n=n,G=G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData)
    } else if(funcType == "Sigmoid")
    {
        sampleDataFromMonotoneSigmoid(n=n,G=G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData)
    } else 
    {
        stop('This function type does not exist!')
    }
}



sampleDataFromGAMGP <- function(n, G, parsFuncType, noiseType, parsNoise, intervMat, intervValues, intervData)
    # INPUTS:   n:  number of samples
    #           G:  adjacency matrix of Graph to simulate from
    #           kap linearly interpolates between GAM and fully nonparametric model. -- kap == 1 --> GAM with additive noise, kap == 0 --> fully nonparametric with additive noise
    #           noiseExp: exponent to model deviations from Gaussian noise -- noiseExp == 1 --> Gaussian noise
    #           intervMat
    #           intervData
    # OUTPUTS:  X:      sampled data    
{
    p <- dim(G)[2]
    X <- matrix(NA,n,p)
    # determine the causal Order which is needed for sampling
    causOrder <- computeCausOrder(G)
    
    if(parsFuncType$output)
    {
        show(causOrder)
    }
    
    # sample noise variances
    noiseVar <- runif(p,parsNoise$varMin,parsNoise$varMax)
    
    # loop through each node according to the causal order
    for(node in causOrder)
    {
        if(parsFuncType$output)
        {
            cat("generating GP for node ", node, "\r")
        }
        paOfNode <- which(G[,node] == 1)
        # simulation of noise at source nodes
        if(length(paOfNode) == 0)
        {
            if(noiseType == "normalRandomVariances" || noiseType == "normalRandomVariancesFixedExp")
            {
                ran <- rnorm(n)
                noisetmp <- (sqrt(noiseVar[node]) * abs(ran))^(parsNoise$noiseExp) * sign(ran)
            } else
            {
                error("This noiseType is not implemented yet.")
            }
            X[,node] <- noisetmp
        } else
        {
            nuPa <- length(paOfNode)
            X[,node] <- rep(0,n)
            
            
            # If kap>0 there is an additive model component
            if(parsFuncType$kap>0)
            {
                for(pa in paOfNode)
                {
                    
                    # sample parameters for Gaussian process 
                    kernPa <- computeGaussKernel(X[,pa],parsFuncType$sigmay,parsFuncType$sigmax)
                    fpa <- mvrnorm(1,rep(0,n),kernPa)
                    if(intervData)
                    {
                        X[intervMat[,node] == FALSE,node] <- X[intervMat[,node] == FALSE,node] + (parsFuncType$kap * fpa)[intervMat[,node] == FALSE]  
                    } else
                    {
                        X[,node] <- X[,node] + parsFuncType$kap * fpa                        
                    }
                }
            }
            
            # if kap<1 there is a non-additive model component
            if(parsFuncType$kap<1)
            {
                kernAllPa <- computeGaussKernel(X[,paOfNode],parsFuncType$sigmay,parsFuncType$sigmax)
                fAllPa <- mvrnorm(1,rep(0,n),kernAllPa)
                if(parsFuncType$output & (parsFuncType$kap==0))
                {
                    ### INCLUDE ADEQUATE PLOTTING FUNCTION (MOREDIMENSIONAL PLOTS) ###                
                }
                if(intervData)
                {
                    X[intervMat[,node] == FALSE,node] <- X[intervMat[,node] == FALSE,node] + ((1-parsFuncType$kap)*fAllPa)[intervMat[,node] == FALSE]
                } else
                {
                    X[,node] <- X[,node] + (1-parsFuncType$kap)*fAllPa 
                }
            }
            
            # Additive noise
            if(noiseType == "normalRandomVariances" || noiseType == "normalRandomVariancesFixedExp")
            {
                ran <- rnorm(n)
                noisetmp <- (0.2*sqrt(noiseVar[node]) * abs(ran))^(parsNoise$noiseExp) * sign(ran)
            } else
            {
                error("This noiseType is not implemented yet.")
            }
            X[,node] <- X[,node] + noisetmp       
        }
    }
    
    return(X)
}





sampleDataFromMonotoneSigmoid <- function(n, G, parsFuncType, noiseType, parsNoise,intervMat,intervValues,intervData)
    # INPUTS:   n:  number of samples
    #           G:  adjacency matrix of Graph to simulate from
    #           
    #           
    # OUTPUTS:  X:      sampled data
    #
    # This function samples from modified (MONOTONE) sigmoid function 
    # c*b*(x+a)/(1+abs(b*(x+a))) where the choice of a,b,c is random.
    
{
    p <- dim(G)[2]
    X <- matrix(NA,n,p)
    # determine the causal Order which is needed for sampling
    causOrder <- computeCausOrder(G)
    
    if(parsFuncType$output)
    {
        show(causOrder)
    }
    
    # sample noise variances
    noiseVar <- runif(p,parsNoise$varMin,parsNoise$varMax)
    
    # loop through each node according to the causal order
    for(node in causOrder)
    {
        if(parsFuncType$output)
        {
            cat("generating GP for node ", node, "\r")
        }
        paOfNode <- which(G[,node] == 1)
        # simulation of noise at source nodes
        if(length(paOfNode) ==0)
        {
            if(noiseType == "normalRandomVariances" || noiseType == "normalRandomVariancesFixedExp")
            {
                ran <- rnorm(n)
                noisetmp <- (sqrt(noiseVar[node]) * abs(ran))^(parsNoise$noiseExp) * sign(ran)
            } else
            {
                error("This noiseType is not implemented yet.")
            }
            X[,node] <- noisetmp
        } else
        {
            nuPa <- length(paOfNode)
            X[,node] <- rep(0,n)
            
            # If kap>0 there is an additive model component
            if(parsFuncType$kap>0)
            {
                for(pa in paOfNode)
                {
                    a.sig <- runif(n=1, min=-2, max=2)
                    bern <- rbinom(1,1,0.5)
                    b.sig <- bern*runif(n=1, min=0.5, max=2) + (1-bern)*runif(n=1, min=-2, max=-0.5)
                    c.sig <- rexp(n=1,rate=4)+1
                    
                    if(intervData)
                    {
                        X[intervMat[,node] == FALSE,node] <- X[intervMat[,node] == FALSE,node] + (c.sig*(b.sig*(X[,pa]+a.sig))/(1+abs(b.sig*(X[,pa]+a.sig))))[intervMat[,node] == FALSE]
                    } else
                    {
                        X[,node] <- X[,node] + c.sig*(b.sig*(X[,pa]+a.sig))/(1+abs(b.sig*(X[,pa]+a.sig)))
                    }
                    if(parsFuncType$output)
                    {
                        plot(X[,pa],c.sig*(b.sig*(X[,pa]+a.sig))/(1+abs(b.sig*(X[,pa]+a.sig))))
                    }
                }
            }
            
            # Additive noise
            if(noiseType == "normalRandomVariances" || noiseType == "normalRandomVariancesFixedExp")
            {
                ran <- rnorm(n)
                noisetmp <- (0.2*sqrt(noiseVar[node]) * abs(ran))^(parsNoise$noiseExp) * sign(ran)
            } else
            {
                error("This noiseType is not implemented yet.")
            }
            X[,node] <- X[,node] + noisetmp       
        }
    }
    
    return(X)
}


sampleDataFromGLinearInterv <- function(n,G,parsFuncType,noiseType,parsNoise,intervMat,intervValues,intervData,changeNoise,boundsForFactorNoiseMultiplication,newBeta)
{
    if(!is.null(parsNoise$noiseVariances))
    {
        parsNoise$noiseVariances <- sqrt(parsNoise$noiseVariances)
    }
    p <- dim(G)[2]
    X <- matrix(NA,n,p)
    
    # determine the causal Order which is needed for sampling
    causOrder <- computeCausOrder(G)
    
    # sample noise variances
    ####??????
    #noiseVar <- runif(p,parsNoise$varMin,parsNoise$varMax)
    
    # loop through each node according to the causal order
    for(node in causOrder)
    {
        X[,node] <- rep(0,n)
        
        # dealing with parents
        paOfNode <- which(G[,node] == 1)
        nuPa <- length(paOfNode)
        if(nuPa > 0)
        {
            # influence of parents
            if(max(is.na(newBeta))==1)
            {
                X[,node] <- X[,node] + X[,paOfNode] %*% matrix(parsFuncType$B[node,paOfNode],nuPa,1)
            } else 
            {
                X[intervMat[,node] == FALSE,node] <- X[intervMat[,node] == FALSE,node] + X[intervMat[,node] == FALSE,paOfNode] %*% matrix(parsFuncType$B[node,paOfNode],nuPa,1)
                if(nuPa ==1)
                {
                    newBetas <- sample(c(-1,1),size=nuPa,0.5)^newBeta[3] * runif(1,newBeta[1],newBeta[2])
                }
                else
                {
                    newBetas <- diag(sample(c(-1,1),size=nuPa,0.5)^newBeta[3]) %*% runif(nuPa,newBeta[1],newBeta[2])
                }
                #warning("remove this")
                #X[intervMat[,node] == TRUE,node] <- X[intervMat[,node] == TRUE,node] + X[intervMat[,node] == TRUE,paOfNode] %*% matrix(rnorm(1,1,0.1)*parsFuncType$B[node,paOfNode],nuPa,1)
                X[intervMat[,node] == TRUE,node] <- X[intervMat[,node] == TRUE,node] + X[intervMat[,node] == TRUE,paOfNode] %*% matrix(newBetas,nuPa,1)                
            }
        }
        
        
        # adding noise
        if(noiseType == "unif")
        {
            noisetmp <- (cbind(parsNoise$bound) %*% rep(1,n)) * matrix(runif(n*1) -0.5, nrow = 1, ncol = n)
        } 
        if(noiseType == "normalRandomVariancesFixedExp")
        {
            noisetmp <- matrix(sign(rnorm(n*1)), nrow = 1, ncol = n) * matrix(rep(runif(1,parsNoise$varMin,parsNoise$varMax),n),nrow = 1, ncol=n) * matrix(abs(rnorm(n*1)), nrow = 1, ncol = n)^(matrix(parsNoise$noiseExp,nrow = 1, ncol=n))
        }
        if(noiseType == "normalRandomVariancesRandomExp")
        {
            # varMin seems to be the minimal stand dev not the min var.
            noisetmp <- matrix(sign(rnorm(n*1)), nrow = 1, ncol = n) * matrix(rep(runif(1,parsNoise$varMin,parsNoise$varMax),n),nrow = 1, ncol=n) * matrix(abs(rnorm(n*1)), nrow = 1, ncol = n)^(matrix(rep(runif(1,parsNoise$noiseExpVarMin,parsNoise$noiseExpVarMax),n),nrow = 1, ncol=n))
        }
        if(noiseType == "normalGivenVariances")
        {
            noisetmp <- rnorm(n, 0, parsNoise$noiseVariances[node])
        }
        if(noiseType == "normalRandomVariances")
        {
            noisetmp <- matrix(rnorm(n*1), nrow = 1, ncol = n) * matrix(rep(runif(1,parsNoise$varMin,parsNoise$varMax),n),nrow = 1, ncol=n)
        }

        if(changeNoise)
        {
            intervSamples <- (intervMat[,node] == TRUE)
            nonIntervSamples <- (intervMat[,node] == FALSE)
            factorsOfNoise <- runif(sum(intervSamples),boundsForFactorNoiseMultiplication[1],boundsForFactorNoiseMultiplication[2])
            X[intervSamples,node] <- X[intervSamples,node] + factorsOfNoise*noisetmp[intervSamples]
            X[nonIntervSamples,node] <- X[nonIntervSamples,node] + noisetmp[nonIntervSamples]
        } else
        {
            X[,node] <- X[,node] + noisetmp       
        }
    }
    
    return(X)
}




sampleDataFromGLinear <- function(n,G,parsFuncType,noiseType,parsNoise)
{
    p <- dim(G)[2]
    Id <- diag(p)
    
    if(noiseType == "unif")
    {
        N <- (cbind(parsNoise$bound) %*% rep(1,n)) * matrix(runif(n*p) -0.5, nrow = p, ncol = n)
        X <- solve(Id-parsFuncType$B) %*% N
        samples <- t(X)   
    } 
    if(noiseType == "normalRandomVariancesFixedExp")
    {
        N <- matrix(sign(rnorm(n*p)), nrow = p, ncol = n) * matrix(rep(runif(p,parsNoise$varMin,parsNoise$varMax),n),nrow = p, ncol=n) * matrix(abs(rnorm(n*p)), nrow = p, ncol = n)^(matrix(parsNoise$noiseExp,nrow = p, ncol=n))
        X <- solve(Id-parsFuncType$B) %*% N
        samples <- t(X)           
    }
    if(noiseType == "normalRandomVariancesRandomExp")
    {
        # varMin seems to be the minimal stand dev not the min var.
        N <- matrix(sign(rnorm(n*p)), nrow = p, ncol = n) * matrix(rep(runif(p,parsNoise$varMin,parsNoise$varMax),n),nrow = p, ncol=n) * matrix(abs(rnorm(n*p)), nrow = p, ncol = n)^(matrix(rep(runif(p,parsNoise$noiseExpVarMin,parsNoise$noiseExpVarMax),n),nrow = p, ncol=n))
        X <- solve(Id-parsFuncType$B) %*% N
        samples <- t(X)          
    }
    if(noiseType == "normalGivenVariances")
    {
        SigmaN <- diag(parsNoise$noiseVariances)
        SigmaX <- solve(Id-parsFuncType$B) %*% SigmaN %*% solve(t(Id - parsFuncType$B))
        samples <- mvrnorm(n, rep(0,p), SigmaX)
        return(samples)
    }
    if(noiseType == "normalRandomVariances")
    {
        N <- matrix(rnorm(n*p), nrow = p, ncol = n) * matrix(rep(runif(p,parsNoise$varMin,parsNoise$varMax),n),nrow = p, ncol=n)
        X <- solve(Id-parsFuncType$B) %*% N
        samples <- t(X)           
    }
    return(samples)
}


computeCausOrder <- function(G)
    # Copyright (c) 2013  Jonas Peters  [peters@stat.math.ethz.ch]
    # All rights reserved.  See the file COPYING for license terms. 
{
    p <- dim(G)[2]
    remaining <- 1:p
    causOrder <- rep(NA,p)
    for(i in 1:(p-1))
    {
        root <- min(which(colSums(G) == 0))
        causOrder[i] <- remaining[root]
        remaining <- remaining[-root]
        G <- G[-root,-root]
    }
    causOrder[p] <- remaining[1]    
    return(causOrder)
}



